#!/usr/bin/perl

#
# $Id: update_DISA_policy_rules.pl,v 1.5 2017/01/30 15:14:21 amanninen Exp $
# upload_policy_rules.pl
#
# Copyright (c)2016, Empowered Networks Inc.
#
# A. Manninen Dec 28, 2016
# amanninen@empowered.ca
#
# Uploads the modified NetMRI Policy and Policy Rule files for DISA. 
#
#

use strict;
use warnings;

use Data::Dumper;
use JSON;
use IO::Socket::SSL;
use LWP;
use LWP::Simple;
use Getopt::Long;
use MIME::Base64;
use File::Basename;

$ENV {'PERL_LWP_SSL_VERIFY_HOSTNAME'} = 0;
$ENV {HTTPS_DEBUG} = 1;

IO::Socket::SSL::set_ctx_defaults(
    SSL_verifycn_scheme => 'www',
    SSL_verify_mode     => 0,
);

# Directory where the files are located.
my $INDIR = glob '~/DISA-STIG/NetMRI';

# CLI input
#
# --host | -h NetMRI host or IP Address.
# --user | -u NetMRI user account string, BASE64 encoded.
# --ci        List of files representing Policies or Policy Rules to be updated
my %options =();
GetOptions(
    "host|h=s" => \$options{h},
    "user|u=s" => \$options{u},
    "ci=s"     => \$options{ci}
);

# NetMRI host and account;
my $user = '';
my $pswd = '';
my $naServer = $options{h} // "192.168.151.34";
my $api = '2.10';
my $files;

if ( $options{ci} ) {
	@$files = split /,/, $options{ci};
}

if ( $options{u} ) {
    ($user, $pswd) = split /:/, $options{u};
}

## URI list

my $baseURI = "https://$naServer";

my $browser = LWP::UserAgent->new;
$browser->cookie_jar({});
$browser->timeout(30);
$browser->env_proxy();
$browser->ssl_opts( verify_hostname => 0 );

$baseURI .= http_post("$baseURI/api/base_uri", {version => $api})->{base_uri}; 

# Logon URI
my $logonURI       = "$baseURI/authenticate";

# Policies URI
my $policiesURI    = "$baseURI/policies";
# Policy Rules URI
my $policyRulesURI = "$baseURI/policy_rules";

my $policies;
my $policyRules;

# Look-up list to find Policy Rules for a Policy
my $fassoPolicies;

# Look-up list to find Policies for a Policy Rule
my $fassoPolicyRules;

###############################################################################
#
# Mainline
#
###############################################################################
print_time("Updating STIG Policies and Policy Rules\n");
http_post($logonURI, {
    username => $user,
    password => $pswd,
}) or exit 0x01;

openListFile();
if ( $files ) {
	checkFiles();
} else {
	getPolicies();
	getPolicyRules();	
}
updateNetMRI();

print_time("STIG Policies and Policy Rules updated.\n");

exit;

###############################################################################
#
# print_time
# Prepend timestamp to print statement
#
###############################################################################
sub print_time{
    
    my $msg = shift;
    print localtime() . " - $msg";
}

###############################################################################
#
# http_post
#
###############################################################################
sub http_post {
    my ($url, $params) = @_;

    my $response = $browser->post($url, $params);

    if ( $response->is_error )
    {
#       my $msg = from_json($response->content);
#       print STDERR "\nFAILED: ".$msg->{message}."\n";
#       for my $field (keys $msg->{fields}) {
#       	print STDERR "*** $field: $_\n" for (@{$msg->{fields}{$field}});
#       }
        return undef;

    }
    else
    {
        my $content = $response->content;

        # This is necessary because NetMRI sends job detail files
        # as bare strings and not in proper JSON format
        eval { $content = from_json($content); 1 };

        return $content;
    }

}

###############################################################################
#
# getPolicies()
#
###############################################################################
sub getPolicies {
	
	print_time("Gathering Policy Files\n");
	
	for (glob("$INDIR/policies/*.json")) {
        
        my $json;
		
        open IN, $_ or print STDERR "Failed to open $_:$!" and exit 0x03;
        while (my $row = <IN>) {
            chomp($row);
	       	$json.=$row;
        }
        close IN;
        
        push @$policies, from_json($json);
        
	} # for each Policy file.
	
} # getPolicies()

###############################################################################
#
# getPolicyRules()
#
###############################################################################
sub getPolicyRules {
	
	print_time("Gathering Policy Rule Files\n");
	
	for (glob("$INDIR/policy_rules/*.json")) {
		
        my $json;
		
        open IN, $_ or print STDERR "Failed to open $_:$!" and exit 0x03;
        while (my $row = <IN>) {
            chomp($row);
	       	$json.=$row;
        }
        close IN;
        
        push @$policyRules, from_json($json);
        
	} # for each Policy Rule file.
	
} # getPolicyRules()

###############################################################################
#
# openListFile()
#
###############################################################################
sub openListFile {
    
    for ( glob "$INDIR/policies/*.list" ) {
        open IN, $_ or print STDERR "Cannot open $_:$!" and exit 0x03;
        while (my $row = <IN>) {
            chomp $row;
            push @{$fassoPolicies->{(substr((basename $_), 0, -5))}}, $row;
            push @{$fassoPolicyRules->{$row}}, substr((basename $_), 0, -5);
        }
        close IN;
        
    }
} # openListFile()

###############################################################################
#
# checkFiles()
#
###############################################################################
sub checkFiles {
	
	print_time("Parsing CLI-supplied files.\n");
	for ( @$files ){
		
		my $json;
		my $rule;
		my ($file) = glob $_;
		
		open IN, $file or print STDERR "Failed to open $file: $!" and exit 0x03;
		while (my $row = <IN>) {
			chomp $row;
			$json.=$row;
			$rule++ if $row =~ m/rule_logic/;
		}
		close IN;
		
#		my $one = ;
		
		push $rule ? \@$policyRules : \@$policies, from_json($json);
		
	}
	
} # checkFiles()

###############################################################################
#
# updateNetMRI()
#
###############################################################################
sub updateNetMRI {
	
	print_time("Updating Policies\n");
	update($_, "policies") for (@$policies);
	
	print_time("Updating Policy Rules\n");
	update($_, "policy_rules") for (@$policyRules);
}

###############################################################################
#
# update()
#
###############################################################################
sub update {
	
	my ($output, $type) = @_;
	
	my $base = $type eq "policies" ? $policiesURI : $policyRulesURI;
	print "$output->{name}...";
	
    my $params = {
        name        => $output->{name},
        short_name  => $output->{short_name},
        description => $output->{description},
        set_filter  => $output->{set_filter},
        author      => $output->{author},
    };

    if ($type eq "policy_rules") {
        $params->{severity}    = $output->{severity};
        $params->{remediation} = $output->{remediation};
        $params->{rule_logic}  = $output->{rule_logic};
    }
    
    my $match = $2 if $output->{name} =~ m/DISA (v\d+, r\d+\.?\d?)?(.*)/;
    # Add escape characters for the following:
    $match =~ s/\(/\\\(/g; # ( -> \(
    $match =~ s/\)/\\\)/g; # ) -> \)
    $match =~ s/\+/\\+/g;  # + -> \+
	
	my $r = http_post("$base/find", { 
		op_name => "rlike",
		val_c_name => "$match\$" 
	});
	
	if ( $r->{total} > 0 ){
		my $ref;
		my $changed = 0;
		if ( $r->{total} == 1){
            $ref = $r->{$type}[0];
		}
		else {
			# Find matching short_name.
			for ( @{$r->{$type}} ) {
				if ($_->{short_name} eq $output->{short_name}) {
					# Short circuit when matched.
					$ref = $_;
					last;
				}
			}
		}
		$changed++ if $ref->{name} ne $output->{name};
		$changed++ if $ref->{short_name} ne $output->{short_name};
		$changed++ if $ref->{description} ne $output->{description};
		$changed++ if $ref->{author} ne $output->{author};
        if ( $ref->{set_filter} and $output->{set_filter}) {
            $changed++ if $ref->{set_filter} ne $output->{set_filter};
		}
        if ($type eq "policy_rules") {
			$changed++ if $ref->{severity} ne $output->{severity};
			$changed++ if $ref->{remediation} ne $output->{remediation};
            $changed++ if $ref->{rule_logic} ne $output->{rule_logic};
		}
        {
			print "Up-to-date" and last unless $changed;
			
			# Update existing.	
			$params->{id} = $ref->{id};
			if (http_post("$base/update", $params) ){
				print "Updated." 
			}
			else {
				print "Failed."
			}
		}
	} # Updated
	else {
		# If not found, create the Policy/Policy Rule
		if ($r = http_post("$base/create", $params)) {
			print "Created.";
			
			if ($type eq "policies") {
				
				# Add all associated Policy Rules to Policy
				for ( @{$fassoPolicies->{"DISA_".($match=~ tr/ /_/)}} ) {
					
					my $policy_rule = http_post({"$policyRulesURI/find",
						op_short_name => '=',
						val_c_short_name => $_
					})->{policy_rules}[0];
										
					print "\nAdded $policy_rule->{name}" if http_post({"$policiesURI/add_policy_rules",
						id => $r->{policy}{id},
						policy_rule_id => $policy_rule->{id}
					});
				}
				
			}
			else {
				
				# Add Policy Rule to all associated Policies.
				for ( @{$fassoPolicyRules->{$output->{short_name}}} ) {
					
					my $find = $_;
					$find = $find =~ s/_/ /gr;
					$find = $find =~ s/DISA //gr;
					
					my $policy = http_post("$policiesURI/find", {
						op_name => 'rlike',
						val_c_name=>  $find,
					})->{policies}[0];                    
					
					print "\nAdded to $policy->{name}" if http_post("$policiesURI/add_policy_rules", {
						id => $policy->{id},
						policy_rule_id => $r->{policy_rule}{id}
					});
				}
			}
			
		} # Created
		else {
			$params->{id} = http_post("$base/find", { 
				val_c_short_name => $params->{short_name},
				op_short_name => '='
			})->{$type}[0]{id};
			if ( $r = http_post("$base/update", $params) ) {
				print "Updated.";
			}
			else {
				print "Error.";
			}
		}
	}
	print "\n";
	
}

# upload_policy_rules.pl
