#!/usr/bin/perl

#
# $Id: prune_old_rules.pl,v 1.1 2017/01/30 15:08:47 amanninen Exp $
# prune_old_rules.pl
#
# Copyright (c)2017, Empowered Networks Inc.
#
# A. Manninen Jan 19, 2017
# amanninen@empowered.ca
#
# Reads a list of NetMRI Policy Rule short_names and prunes the Policy Rules.
#
# 

use strict;
use warnings;

use Data::Dumper;
use JSON;
use IO::Socket::SSL;
use LWP;
use LWP::Simple;
use Getopt::Long;
use MIME::Base64;

$ENV{'PERL_LWP_SSL_VERIFY_HOSTNAME'} = 0;
$ENV{HTTPS_DEBUG} = 1;

# List of rules that are to be deleted
my $rules2Del = glob '~/DISA-STIG/NetMRI/rules2Del';

IO::Socket::SSL::set_ctx_defaults(
	SSL_verifycn_scheme => 'www',
	SSL_verify_mode => 0,
);

my %opt = ();
GetOptions(
	"host|h=s" => \$opt{h},
	"user|u=s" => \$opt{u}
);

# NetMRI host and account
my $naServer = $opt{h} //'192.168.151.34';
my $username = '';
my $passwd = '';

if ( $opt{u} ) {
	($username, $passwd) = split /:/, $opt{u};
}

# NetMRI API version to use
my $apiVersion = '2.10';

# base URI
my $baseUri = "https://$naServer";

my $browser = LWP::UserAgent->new;
$browser->cookie_jar({});
$browser->timeout(30);
$browser->env_proxy;
$browser->ssl_opts( verify_hostname => 0 );

$baseUri .= http_post("$baseUri/api/base_uri", { version => $apiVersion })->{base_uri};
# Logon URL
my $logonUrl      = "$baseUri/authenticate";
my $policyRuleURL = "$baseUri/policy_rules";
my $policiesURL   = "$baseUri/policies";

my @rules = ();
my $policies;

my $EXITCODE = 0x00;

###############################################################################
#
# Mainline
#
###############################################################################
http_post($logonUrl,{
	username => $username,
	password => $passwd
}) or exit 0x01;

openFile();
getPolicies();
print localtime() . " - Pruning old DISA STIG Policy Rules.";
for my $rule (@rules) { delRule($rule) }

exit $EXITCODE;

###############################################################################
#
# http_post
#
###############################################################################
sub http_post {
	
	my ($url, $params) = @_;
	
	if ( (my $r = $browser->post($url, $params)) -> is_error ) {
		print STDERR "ERROR: ".$r->status_line."\n";
		print STDERR "URL: $url\n";
		return undef;
	}
	else {
		my $content = $r->content;
		eval { $content = from_json($content); 1 };
		return $content;
	}
	
} # http_post()

###############################################################################
#
# openFile
#
###############################################################################
sub openFile {
	
	open IN, $rules2Del or die 0x03;
	while (<IN>) {
		chomp $_;
		push @rules, $_;
	}
	close IN;
		
} # openFile()

###############################################################################
#
# getPolicies
#
###############################################################################
sub getPolicies {
	
	my $r = http_post("$policiesURL/find", {
		op_name => "rlike",
		val_c_name => "^DISA v8"
	});
	
	for (@{$r->{policies}}) {
		$policies->{$_->{short_name}} = {
			id => $_->{id},
			name => $_->{name}
		};
	}
	
} # getPolicies()

###############################################################################
#
# delRule
#
###############################################################################
sub delRule {
	
	my $rule = shift;
	
	print "$rule";
	my $policy_rule_id = http_post("$policyRuleURL/find", {
		op_short_name => '=',
		val_c_short_name => $rule		
	})->{policy_rules}[0]{id};
	
	## If there is no id, the rule was previously removed.
	# Return from subroutine.
	if ( ! $policy_rule_id ) {
		print "...Not found. Skipping.\n";
		return;
	}
	
	## Remove the Policy Rule from Policies.
	for my $policy (keys $policies) {
		
#		print Dumper $policies->{$policy};
		
		my $pr = http_post("$policiesURL/policy_rules", { 
            id => $policies->{$policy}{id}
        })->{policy_rules};
        				
		for my $policy_rule ( @{$pr} ) {
			
			if ( $policy_rule->{id} = $policy_rule_id ) {
				print "\n*** Removing $rule from $policy...";
				print "Removed" if http_post(
				"$policiesURL/remove_policy_rules", {
					id => $policies->{$policy}{id},
					policy_rule_id => $policy_rule_id
				});
				last; # break the loop once a match is found.
			}
			
		} # foreach policy rule in policy
		
	} # for each policy
	
	print "\nDeleting $rule...";
	if ( http_post("$policyRuleURL/destroy", {id => $policy_rule_id}) ) {
		print "Deleted."
	}
	else {
		print "Error.";
		$EXITCODE = 0x04;
	}
	print "\n";
	
} # delRule()

# prune_old_rules.pl
